'use client'

import { useState, useEffect } from 'react'

export default function InsuranceCalculator() {
  const [formData, setFormData] = useState({
    age: '',
    gender: '',
    vehicleYear: '',
    vehicleValue: '',
    location: '',
    drivingRecord: '',
    coverage: '',
    deductible: ''
  })
  const [estimatedCost, setEstimatedCost] = useState<number | null>(null)
  const [isCalculating, setIsCalculating] = useState(false)

  const calculatePremium = () => {
    setIsCalculating(true)

    // Simulate calculation delay
    setTimeout(() => {
      let baseCost = 800 // Base annual premium

      // Age factor
      const age = parseInt(formData.age)
      if (age < 25) baseCost *= 1.5
      else if (age < 30) baseCost *= 1.2
      else if (age > 65) baseCost *= 1.1

      // Vehicle value factor
      const vehicleValue = parseInt(formData.vehicleValue)
      if (vehicleValue > 50000) baseCost *= 1.4
      else if (vehicleValue > 30000) baseCost *= 1.2
      else if (vehicleValue < 10000) baseCost *= 0.8

      // Vehicle year factor
      const vehicleYear = parseInt(formData.vehicleYear)
      const currentYear = new Date().getFullYear()
      const vehicleAge = currentYear - vehicleYear
      if (vehicleAge < 3) baseCost *= 1.2
      else if (vehicleAge > 10) baseCost *= 0.9

      // Driving record factor
      switch (formData.drivingRecord) {
        case 'excellent': baseCost *= 0.8; break
        case 'good': baseCost *= 1; break
        case 'fair': baseCost *= 1.3; break
        case 'poor': baseCost *= 1.8; break
      }

      // Coverage level factor
      switch (formData.coverage) {
        case 'basic': baseCost *= 0.7; break
        case 'standard': baseCost *= 1; break
        case 'premium': baseCost *= 1.4; break
        case 'comprehensive': baseCost *= 1.8; break
      }

      // Deductible factor
      switch (formData.deductible) {
        case '250': baseCost *= 1.2; break
        case '500': baseCost *= 1; break
        case '1000': baseCost *= 0.85; break
        case '2500': baseCost *= 0.7; break
      }

      // Location factor
      switch (formData.location) {
        case 'urban': baseCost *= 1.3; break
        case 'suburban': baseCost *= 1; break
        case 'rural': baseCost *= 0.8; break
      }

      setEstimatedCost(Math.round(baseCost))
      setIsCalculating(false)
    }, 1500)
  }

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault()
    calculatePremium()
  }

  const handleChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    setFormData(prev => ({
      ...prev,
      [e.target.name]: e.target.value
    }))
    setEstimatedCost(null)
  }

  const isFormValid = Object.values(formData).every(value => value !== '')

  return (
    <div className="bg-white rounded-lg shadow-lg p-8">
      <div className="text-center mb-8">
        <h3 className="text-3xl font-bold text-gray-900 mb-4">Auto Insurance Calculator</h3>
        <p className="text-gray-600">Get an estimated quote for your auto insurance premium</p>
      </div>

      <form onSubmit={handleSubmit} className="space-y-6">
        <div className="grid md:grid-cols-2 gap-6">
          <div>
            <label htmlFor="age" className="block text-sm font-medium text-gray-700 mb-2">
              Your Age
            </label>
            <select
              id="age"
              name="age"
              value={formData.age}
              onChange={handleChange}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            >
              <option value="">Select Age Range</option>
              <option value="18">18-24</option>
              <option value="25">25-29</option>
              <option value="30">30-39</option>
              <option value="40">40-49</option>
              <option value="50">50-64</option>
              <option value="65">65+</option>
            </select>
          </div>

          <div>
            <label htmlFor="gender" className="block text-sm font-medium text-gray-700 mb-2">
              Gender
            </label>
            <select
              id="gender"
              name="gender"
              value={formData.gender}
              onChange={handleChange}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            >
              <option value="">Select Gender</option>
              <option value="male">Male</option>
              <option value="female">Female</option>
              <option value="other">Other</option>
            </select>
          </div>
        </div>

        <div className="grid md:grid-cols-2 gap-6">
          <div>
            <label htmlFor="vehicleYear" className="block text-sm font-medium text-gray-700 mb-2">
              Vehicle Year
            </label>
            <select
              id="vehicleYear"
              name="vehicleYear"
              value={formData.vehicleYear}
              onChange={handleChange}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            >
              <option value="">Select Year</option>
              {Array.from({ length: 25 }, (_, i) => 2025 - i).map(year => (
                <option key={year} value={year}>{year}</option>
              ))}
            </select>
          </div>

          <div>
            <label htmlFor="vehicleValue" className="block text-sm font-medium text-gray-700 mb-2">
              Vehicle Value
            </label>
            <select
              id="vehicleValue"
              name="vehicleValue"
              value={formData.vehicleValue}
              onChange={handleChange}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            >
              <option value="">Select Value Range</option>
              <option value="5000">Under $10,000</option>
              <option value="15000">$10,000 - $20,000</option>
              <option value="25000">$20,000 - $30,000</option>
              <option value="40000">$30,000 - $50,000</option>
              <option value="75000">$50,000+</option>
            </select>
          </div>
        </div>

        <div className="grid md:grid-cols-2 gap-6">
          <div>
            <label htmlFor="location" className="block text-sm font-medium text-gray-700 mb-2">
              Location Type
            </label>
            <select
              id="location"
              name="location"
              value={formData.location}
              onChange={handleChange}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            >
              <option value="">Select Location</option>
              <option value="urban">Urban (City)</option>
              <option value="suburban">Suburban</option>
              <option value="rural">Rural</option>
            </select>
          </div>

          <div>
            <label htmlFor="drivingRecord" className="block text-sm font-medium text-gray-700 mb-2">
              Driving Record
            </label>
            <select
              id="drivingRecord"
              name="drivingRecord"
              value={formData.drivingRecord}
              onChange={handleChange}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            >
              <option value="">Select Record</option>
              <option value="excellent">Excellent (No violations)</option>
              <option value="good">Good (1 minor violation)</option>
              <option value="fair">Fair (2-3 violations)</option>
              <option value="poor">Poor (Multiple violations)</option>
            </select>
          </div>
        </div>

        <div className="grid md:grid-cols-2 gap-6">
          <div>
            <label htmlFor="coverage" className="block text-sm font-medium text-gray-700 mb-2">
              Coverage Level
            </label>
            <select
              id="coverage"
              name="coverage"
              value={formData.coverage}
              onChange={handleChange}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            >
              <option value="">Select Coverage</option>
              <option value="basic">Basic (State Minimum)</option>
              <option value="standard">Standard (Recommended)</option>
              <option value="premium">Premium (Enhanced)</option>
              <option value="comprehensive">Comprehensive (Full)</option>
            </select>
          </div>

          <div>
            <label htmlFor="deductible" className="block text-sm font-medium text-gray-700 mb-2">
              Deductible Amount
            </label>
            <select
              id="deductible"
              name="deductible"
              value={formData.deductible}
              onChange={handleChange}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            >
              <option value="">Select Deductible</option>
              <option value="250">$250</option>
              <option value="500">$500</option>
              <option value="1000">$1,000</option>
              <option value="2500">$2,500</option>
            </select>
          </div>
        </div>

        <button
          type="submit"
          disabled={!isFormValid || isCalculating}
          className="w-full bg-blue-600 text-white py-4 px-6 rounded-lg font-semibold hover:bg-blue-700 focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
        >
          {isCalculating ? (
            <span className="flex items-center justify-center">
              <svg className="animate-spin -ml-1 mr-3 h-5 w-5 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
              </svg>
              Calculating...
            </span>
          ) : (
            'Calculate My Premium'
          )}
        </button>
      </form>

      {estimatedCost && (
        <div className="mt-8 bg-green-50 border border-green-200 rounded-lg p-6">
          <div className="text-center">
            <h4 className="text-2xl font-bold text-green-800 mb-2">Estimated Annual Premium</h4>
            <div className="text-4xl font-bold text-green-600 mb-4">${estimatedCost.toLocaleString()}</div>
            <p className="text-green-700 mb-4">This is an estimate based on the information provided</p>
            <div className="space-y-2 text-sm text-green-600">
              <p>• Monthly payment: ~${Math.round(estimatedCost / 12)}</p>
              <p>• Semi-annual payment: ~${Math.round(estimatedCost / 2)}</p>
            </div>
            <button className="mt-4 bg-green-600 text-white px-6 py-2 rounded-lg hover:bg-green-700 transition-colors">
              Get Actual Quote
            </button>
          </div>
        </div>
      )}

      <div className="mt-6 text-xs text-gray-500 text-center">
        <p>* This calculator provides estimates only. Actual premiums may vary based on additional factors and carrier underwriting guidelines.</p>
      </div>
    </div>
  )
}
